<?php
include "../../root.class.php";

function calculateNetSalaryWithDbTax($grossMonthlySalary)
{
    $db = new db_safeguard();
    $today = new DateTime();

    // 1. Load all tax years from DB and determine which one today falls into
    $tax_years_res = $db->select_query("tax_years", "*", "1 ORDER BY year DESC");

    $tax_year = null;

    while ($row = $tax_years_res->fetch_assoc()) {
        $start = DateTime::createFromFormat("Y-m-d", ($row['year'] - 1) . "-03-01");
        $end = DateTime::createFromFormat("Y-m-d", $row['year'] . "-02-28");

        // Handle leap years
        if ((int) $row['year'] % 4 == 0) {
            $end = DateTime::createFromFormat("Y-m-d", $row['year'] . "-02-29");
        }

        if ($today >= $start && $today <= $end) {
            $tax_year = $row;
            break;
        }
    }

    if (!$tax_year) {
        return ['error' => "No tax year found for today (" . $today->format('Y-m-d') . ")"];
    }

    $annualIncome = $grossMonthlySalary * 12;

    // 2. Get the correct tax bracket
    $brackets_res = $db->select_query("tax_brackets", "*", "tax_year_id = {$tax_year['id']} ORDER BY income_from ASC");
    $bracket = null;

    while ($b = $brackets_res->fetch_assoc()) {
        $incomeFrom = (float) $b['income_from'];
        $incomeTo = isset($b['income_to']) && is_numeric($b['income_to']) ? (float) $b['income_to'] : INF;

        if ($annualIncome >= $incomeFrom && $annualIncome <= $incomeTo) {
            $bracket = $b;
            break;
        }
    }

    if (!$bracket) {
        return ['error' => "No matching tax bracket found for income R$annualIncome"];
    }

    // 3. Calculate tax before rebate
    $base_tax = $bracket['base_tax'];
    $tax_rate = $bracket['tax_rate'];
    $excess_income = $annualIncome - $bracket['income_from'];
    $annual_tax = $base_tax + ($excess_income * $tax_rate);

    // 4. Subtract primary rebate
    $annual_tax -= $tax_year['primary_rebate'];
    $annual_tax = max($annual_tax, 0); // Cannot be negative

    // 5. Monthly breakdown
    $monthly_tax = $annual_tax / 12;
    $uif = min($grossMonthlySalary * $tax_year['uif_rate'], $tax_year['uif_cap']);
    $net = $grossMonthlySalary - $monthly_tax - $uif;

    return [
        'monthly_tax' => round($monthly_tax, 2),
        'uif' => round($uif, 2),
        'net_monthly_salary' => round($net, 2),
        'annual_taxable_income' => round($annualIncome, 2),
        'annual_tax' => round($annual_tax, 2),
        'tax_year' => $tax_year['year']
    ];
}


$result = calculateNetSalaryWithDbTax(30000);

if (isset($result['error'])) {
    echo "Error: " . $result['error'];
} else {
    echo "Monthly Tax: R" . $result['monthly_tax'] . "\n";
    echo "UIF: R" . $result['uif'] . "\n";
    echo "Net Monthly Salary: R" . $result['net_monthly_salary'] . "\n";
}
